<?php

namespace App\Http\Controllers;

use App\Models\Course;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class CourseController extends Controller
{
    public function index(): JsonResponse
    {
        $courses = Course::with('language', 'category')->get();
        return response()->json([
            'status' => 'success',
            'data' => $courses,
        ], 200);
    }



    public function show($id): JsonResponse
    {
        $course = Course::with('language', 'category')->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $course,
        ], 200);
    }

 public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Title' => 'required|string|max:255',
            'Description' => 'nullable|string',
            'Content_url' => 'nullable|file|mimes:pdf,doc,docx|max:102400',
            'Is_paid' => 'sometimes|boolean',
            'Price' => 'sometimes|numeric|min:0',
            'Language_id' => 'sometimes|nullable|exists:Languages,Language_id',
            'Category_id' => 'sometimes|nullable|exists:course_categories,Category_id',
            'Status' => 'sometimes|in:draft,active,inactive,archived',
            'Pass_threshold' => 'sometimes|integer|min:0|max:100',
            'Exam_Question' => 'sometimes|integer|min:0',
        ], [
            'Content_url.file' => 'The course content must be a valid file.',
            'Content_url.mimes' => 'The course content must be a PDF, DOC, or DOCX file.',
            'Content_url.max' => 'The course content must not exceed 100MB.',
            'Pass_threshold.integer' => 'The pass threshold must be an integer.',
            'Pass_threshold.min' => 'The pass threshold must be at least 0.',
            'Pass_threshold.max' => 'The pass threshold must not exceed 100.',
            'Exam_Question.integer' => 'The exam question count must be an integer.',
            'Exam_Question.min' => 'The exam question count must be at least 0.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'Title',
            'Description',
            'Is_paid',
            'Price',
            'Language_id',
            'Category_id',
            'Status',
            'Pass_threshold',
            'Exam_Question',
        ]);

        // Set default values
        $data['Is_paid'] = $data['Is_paid'] ?? 0; // Default to false (free)
        $data['Status'] = $data['Status'] ?? 'active';
        $data['Pass_threshold'] = $data['Pass_threshold'] ?? 70;
        $data['Exam_Question'] = $data['Exam_Question'] ?? 0;

        if ($request->hasFile('Content_url') && $request->file('Content_url')->isValid()) {
            $content = $request->file('Content_url');
            $contentExtension = $content->getClientOriginalExtension();
            $contentName = time() . '_content.' . $contentExtension;
            $contentPath = 'public/course_contents/' . $contentName;
            Storage::putFileAs('public/course_contents', $content, $contentName);
            $data['Content_url'] = url('storage/course_contents/' . $contentName);
        } elseif ($request->has('Content_url')) {
            $contentInput = $request->input('Content_url');
            if (is_string($contentInput) && filter_var($contentInput, FILTER_VALIDATE_URL) && strpos($contentInput, url('storage/course_contents/')) === 0) {
                $data['Content_url'] = $contentInput;
            } else {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Course content must be a valid storage URL',
                ], 422);
            }
        }

        $course = Course::create($data);

        return response()->json([
            'status' => 'success',
            'data' => $course,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $course = Course::find($id);
        if (!$course) {
            Log::warning("Course not found for update", [
                'course_id' => $id,
            ]);
            return response()->json([
                'status' => 'error',
                'message' => 'Course not found',
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'Title' => 'sometimes|string|max:255',
            'Description' => 'sometimes|nullable|string',
            'Content_url' => 'sometimes|nullable|string',
            'Is_paid' => 'sometimes|boolean',
            'Price' => 'sometimes|numeric|min:0',
            'Language_id' => 'sometimes|nullable|exists:Languages,Language_id',
            'Category_id' => 'sometimes|nullable|exists:course_categories,Category_id',
            'Status' => 'sometimes|in:draft,active,inactive,archived',
            'Pass_threshold' => 'sometimes|integer|min:0|max:100',
            'Exam_Question' => 'sometimes|integer|min:0',
        ], [
            'Pass_threshold.integer' => 'The pass threshold must be an integer.',
            'Pass_threshold.min' => 'The pass threshold must be at least 0.',
            'Pass_threshold.max' => 'The pass threshold must not exceed 100.',
            'Exam_Question.integer' => 'The exam question count must be an integer.',
            'Exam_Question.min' => 'The exam question count must be at least 0.',
        ]);

        if ($validator->fails()) {
            Log::warning("Validation failed for course update", [
                'course_id' => $id,
                'errors' => $validator->errors()->toArray()
            ]);
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'Title',
            'Description',
            'Is_paid',
            'Price',
            'Language_id',
            'Category_id',
            'Status',
            'Pass_threshold',
            'Exam_Question',
        ]);

        $validMimes = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];

        if ($request->has('Content_url') && !empty($request->input('Content_url'))) {
            $contentInput = $request->input('Content_url');
            if ($this->isValidBase64($contentInput)) {
                $mimeType = $this->getMimeTypeFromBase64($contentInput);
                if (!in_array($mimeType, $validMimes)) {
                    return response()->json([
                        'status' => 'error',
                        'message' => "Invalid MIME type: $mimeType, expected: " . implode(', ', $validMimes),
                    ], 422);
                }

                $base64String = preg_replace('/^data:application\/[\w\.-]+;base64,/', '', $contentInput);
                $contentData = base64_decode($base64String, true);
                if ($contentData === false) {
                    Log::warning("Failed to decode base64 string for Content_url in course update", [
                        'course_id' => $id,
                        'content_input' => substr($contentInput, 0, 100) . '...'
                    ]);
                    $data['Content_url'] = $course->Content_url;
                } elseif (strlen($contentData) > 102400 * 1024) {
                    Log::warning("Content_url file size exceeds 100MB in course update", [
                        'course_id' => $id,
                        'size' => strlen($contentData)
                    ]);
                    $data['Content_url'] = $course->Content_url;
                } else {
                    if ($course->Content_url) {
                        $oldContentPath = str_replace(url('storage/'), 'public/', $course->Content_url);
                        if (Storage::exists($oldContentPath)) {
                            Storage::delete($oldContentPath);
                            Log::info("Deleted old content file for course", [
                                'course_id' => $id,
                                'old_content_path' => $oldContentPath
                            ]);
                        }
                    }

                    $contentExtension = [
                        'application/pdf' => 'pdf',
                        'application/msword' => 'doc',
                        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx'
                    ][$mimeType];
                    $contentName = time() . '_content.' . $contentExtension;
                    $contentPath = 'public/course_contents/' . $contentName;
                    Storage::put($contentPath, $contentData);
                    $data['Content_url'] = url('storage/course_contents/' . $contentName);
                    Log::info("Successfully updated Content_url with base64 for course", [
                        'course_id' => $id,
                        'new_content_path' => $contentPath
                    ]);
                }
            } elseif (filter_var($contentInput, FILTER_VALIDATE_URL) && strpos($contentInput, url('storage/course_contents/')) === 0) {
                if ($course->Content_url && $course->Content_url !== $contentInput) {
                    $oldContentPath = str_replace(url('storage/'), 'public/', $course->Content_url);
                    if (Storage::exists($oldContentPath)) {
                        Storage::delete($oldContentPath);
                        Log::info("Deleted old content file for course due to new URL", [
                            'course_id' => $id,
                            'old_content_path' => $oldContentPath
                        ]);
                    }
                }
                $data['Content_url'] = $contentInput;
                Log::info("Updated Content_url with valid storage URL for course", [
                    'course_id' => $id,
                    'content_url' => $contentInput
                ]);
            } else {
                Log::warning("Invalid Content_url provided in course update", [
                    'course_id' => $id,
                    'content_input' => substr($contentInput, 0, 100) . '...'
                ]);
                $data['Content_url'] = $course->Content_url;
            }
        } elseif ($request->hasFile('Content_url') && $request->file('Content_url')->isValid()) {
            if ($course->Content_url) {
                $oldContentPath = str_replace(url('storage/'), 'public/', $course->Content_url);
                if (Storage::exists($oldContentPath)) {
                    Storage::delete($oldContentPath);
                    Log::info("Deleted old content file for course due to new file upload", [
                        'course_id' => $id,
                        'old_content_path' => $oldContentPath
                    ]);
                }
            }

            $content = $request->file('Content_url');
            $fileMime = $content->getMimeType();
            if (!in_array($fileMime, $validMimes)) {
                return response()->json([
                    'status' => 'error',
                    'message' => "Invalid MIME type: $fileMime, expected: " . implode(', ', $validMimes),
                ], 422);
            }
            $contentExtension = $content->getClientOriginalExtension();
            $contentName = time() . '_content.' . $contentExtension;
            $contentPath = 'public/course_contents/' . $contentName;
            Storage::putFileAs('public/course_contents', $content, $contentName);
            $data['Content_url'] = url('storage/course_contents/' . $contentName);
            Log::info("Successfully updated Content_url with file upload for course", [
                'course_id' => $id,
                'new_content_path' => $contentPath
            ]);
        }

        if (!empty($data)) {
            $course->update($data);
        }

        return response()->json([
            'status' => 'success',
            'data' => $course->fresh(),
            'message' => isset($contentInput) && !empty($contentInput) && isset($data['Content_url']) && $data['Content_url'] === $course->Content_url
                ? 'Course updated, invalid content URL or base64 string was skipped'
                : 'Course updated successfully',
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $course = Course::findOrFail($id);

        if ($course->Content_url) {
            $contentPath = str_replace(url('storage/'), 'public/', $course->Content_url);
            Storage::delete($contentPath);
        }

        $course->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Course deleted successfully',
        ], 200);
    }

   public function courseCounts(): JsonResponse
    {
        $totalCourses = Course::count();
        $activeCourses = Course::where('Status', 'active')->count();
        $closedCourses = Course::where('Status', 'closed')->count();

        return response()->json([
            'status' => 'success',
            'data' => [
                'total_courses' => $totalCourses,
                'active_courses' => $activeCourses,
                'closed_courses' => $closedCourses,
            ],
        ], 200);
    }

    private function getMimeTypeFromBase64($base64String)
    {
        if (preg_match('/^data:application\/([\w\.-]+);base64,/', $base64String, $matches)) {
            return 'application/' . $matches[1];
        }

        $base64Data = preg_replace('/^data:application\/[\w\.-]+;base64,/', '', $base64String);
        $decodedData = base64_decode($base64Data, true);

        if ($decodedData === false) {
            return 'invalid';
        }

        if (substr($decodedData, 0, 4) === '%PDF') {
            return 'application/pdf';
        }

        return 'application/octet-stream';
    }

    private function isValidBase64($string)
    {
        $base64Data = preg_replace('/^data:application\/[\w\.-]+;base64,/', '', $string);
        if (!preg_match('/^[A-Za-z0-9+\/=]+$/', $base64Data)) {
            return false;
        }
        if (strlen($base64Data) % 4 !== 0) {
            return false;
        }
        return true;
    }


}

